//=============================================================================
// RPGツクールMZ - AltBattleScreen.js
//-----------------------------------------------------------------------------
// このプラグインは以下のプラグインを参考に作成されました。
//
// ルルの境界 (Lulu's Church) 様
// LL_StandingPictureBattle.js
//=============================================================================

/*:ja
 * @target MZ
 * @plugindesc 戦闘画面のレイアウトを変更します。
 * @author emoriiin979
 *
 * @help AltBattleScreen.js
 *
 * 戦闘画面に立ち絵などを表示します。
 *
 * 下記のタイミングで立ち絵を更新します。
 *   ・戦闘開始時 (戦う・逃げる選択前)
 *   ・被ダメージ時
 *
 * 画像ファイルの表示優先順:
 *   1. ステートID、スイッチID、変数条件全てに一致するもの
 *   2. ステートID、スイッチID両方に一致するもの
 *   3. ステートID、変数条件両方に一致するもの
 *   4. ステートIDのみ一致するもの
 *   5. スイッチID、変数条件両方に一致するもの
 *   6. スイッチIDのみ一致するもの
 *   7. 変数条件のみ一致するもの
 *   8. 条件なし (ステートID、スイッチID、変数条件全て設定なし)
 *
 * @param sPictureSettings
 * @text 立ち絵リスト
 * @desc ステート、スイッチ、変数条件ごとに立ち絵を複数定義できます。
 * ※この項目は使用しません
 *
 * @param sCommandPictures
 * @text コマンド選択時
 * @desc コマンド選択中に表示する立ち絵を定義します。
 * @default []
 * @type struct<sPictures>[]
 * @parent sPictureSettings
 *
 * @param sDamagePictures
 * @text ダメージ時
 * @desc 被ダメージ時に表示する立ち絵を定義します。
 * @default []
 * @type struct<sPictures>[]
 * @parent sPictureSettings
 *
 * @param sTurnEndPictures
 * @text ターン終了時
 * @desc ターン終了時のイベント中に表示する立ち絵を定義します。
 * @default []
 * @type struct<sPictures>[]
 * @parent sPictureSettings
 *
 * @param gaugeWidth
 * @text ゲージの幅
 * @desc 戦闘画面のゲージ幅を定義します。
 * RPGツクールMZでのデフォルト値は128です。
 * @default 256
 * @type number
 *
 * @command performDamageEx
 * @text ダメージ表現EX
 * @desc 既存のダメージ表現処理を拡張しました。
 *
 * @arg damageType
 * @text ダメージ種類
 * @desc ダメージの種類を設定します。
 * @default normal
 * @type select
 * @option 通常
 * @value normal
 * @option 強力
 * @value hard
 * @option 無
 * @value no
 *
 * @arg changeSPicture
 * @text 立ち絵変更の有無
 * @desc 立ち絵変更の有無を選択します。
 * @default 1
 * @type select
 * @option 有り
 * @value 1
 * @option 無し
 * @value 0
 *
 * @arg damageTarget
 * @text ダメージ対象
 * @desc ダメージの対象(HP/TP)を指定します。
 * @default hp
 * @type select
 * @option HP
 * @value hp
 * @option TP
 * @value tp
 * @option 無し
 * @value no
 *
 * @arg damageValue
 * @text ダメージ値
 * @desc 与えるダメージを数値/数式で指定します。
 * this.x : アクター各種値、d(N) : 分散値
 * @default 0
 *
 * @command changeSPicture
 * @text 立ち絵変更
 * @desc 立ち絵を強制的に変更します。
 *
 * @arg sPictureType
 * @text 立ち絵種別
 * @desc どの立ち絵に変更するか選択します。
 * @default command
 * @type select
 * @option コマンド選択
 * @value command
 * @option 被ダメージ
 * @value damage
 * @option 絶頂
 * @value orgasm
 *
 * @command setRandomId
 * @text ランダムID設定
 * @desc 現在のランダムIDを強制的に変更できます。
 * ここで設定されたIDが選択肢から「除外される」ことに注意してください。
 *
 * @arg randomId
 * @text ランダムID
 * @desc 設定するランダムIDを指定してください。
 * @type number
 */

/*~struct~sPictures:
 *
 * @param imageName
 * @text 画像ファイル名
 * @desc 立ち絵として表示する画像ファイルを選択してください。
 * @dir img/pictures
 * @type file
 * @require 1
 *
 * @param actorId
 * @text アクターID
 * @desc アクターIDです。立ち絵を定義するアクターを選択してください。
 * @default 1
 * @type actor
 *
 * @param stateCase
 * @text ステート条件
 * @desc ステート条件で立ち絵を変更したい場合に使用します。
 * @default []
 * @type struct<stateCase>[]
 *
 * @param switchCase
 * @text スイッチ条件
 * @desc スイッチ条件で立ち絵を変更したい場合に使用します。
 * @default []
 * @type struct<switchCase>[]
 *
 * @param variableCase
 * @text 変数条件
 * @desc 変数条件で立ち絵を変更したい場合に使用します。
 * @default []
 * @type struct<variableCase>[]
 *
 * @param origin
 * @text 原点
 * @desc 立ち絵の原点です。
 * @default upperleft
 * @type select
 * @option 左上
 * @value upperleft
 * @option 中央
 * @value center
 *
 * @param x
 * @text X座標
 * @desc 立ち絵の表示位置(X)です。
 * @default 648
 * @min -9999
 * @max 9999
 * @type number
 *
 * @param y
 * @text Y座標
 * @desc 立ち絵の表示位置(Y)です。
 * @default 20
 * @min -9999
 * @max 9999
 * @type number
 *
 * @param scaleX
 * @text X拡大率
 * @desc 立ち絵の拡大率(X)です。
 * @default 100
 * @min -2000
 * @max 2000
 * @type number
 *
 * @param scaleY
 * @text Y拡大率
 * @desc 立ち絵の拡大率(Y)です。
 * @default 100
 * @min -2000
 * @max 2000
 * @type number
 *
 * @param motion
 * @text モーション
 * @desc 再生モーションを選択してください。
 * @default floatrightfast
 * @type select
 * @option なし
 * @value none
 * @option 右からフロートイン (コマンド)
 * @value floatrightfast
 * @option 左からフロートイン (コマンド)
 * @value floatleftfast
 * @option 頷く
 * @value yes
 * @option ジャンプ
 * @value jump
 * @option 繰り返しジャンプ
 * @value jumploop
 * @option ガクガクし続ける
 * @value shakeloop
 * @option 横に揺れ続ける
 * @value noslowloop
 * @option 息づかい風
 * @value breathing
 * @option 息づかい風 (伸縮)
 * @value breathing2
 * @option 揺れる (ダメージ)
 * @value damage
 * @option 超揺れる（大ダメージ）
 * @value harddamage
 * @option 右からフロートイン (勝利)
 * @value floatright
 * @option 左からフロートイン (勝利)
 * @value floatleft
 * @option 左へスライド (攻撃)
 * @value stepleft
 * @option 右へスライド (攻撃)
 * @value stepright
 * @option 頭を下げる (防御)
 * @value headdown
 *
 * @param rating
 * @text 優先度
 * @desc 立ち絵選択の優先度を1(最小)～9(最大)で指定します。
 * @default 5
 * @min 1
 * @max 9
 * @type number
 */

/*~struct~stateCase:
 *
 * @param id
 * @text ステートID
 * @desc 条件に使用するステートIDです。
 * @type state
 */

/*~struct~switchCase:
 *
 * @param id
 * @text スイッチID
 * @desc 条件に使用するスイッチIDです。
 * @type switch
 */

/*~struct~variableCase:
 *
 * @param id
 * @text 変数ID
 * @desc 条件に使用する変数IDです。
 * @type variable
 *
 * @param type
 * @text 変数条件
 * @desc 変数IDとの比較条件です。
 * @default equal
 * @type select
 * @option 一致
 * @value equal
 * @option 以上
 * @value higher
 * @option 以下
 * @value lower
 *
 * @param value
 * @text 変数比較数値
 * @desc 変数IDと比較する数値です。
 * @default 0
 * @min -99999999
 * @max 99999999
 * @type number
 */

(() => {
    "use strict";

    const pluginName = "AltBattleScreen";
    const parameters = PluginManager.parameters(pluginName);
    
    // 独自変数定義
    let commandActorId = null;
    let damageActorId = null;
    let orgasmActorId = null;

    //=========================================================================
    // プラグインコマンド
    //=========================================================================

    // ダメージ表現EX
    PluginManager.registerCommand(pluginName, "performDamageEx", args => {
        const actor = $gameActors.actor(1);
        const changeSPictureFlag = ! (parseInt(args["changeSPicture"]) === 0);
        actor.performDamageEx(args["damageType"], args["damageTarget"], args["damageValue"], changeSPictureFlag);
    });

    // 立ち絵変更
    PluginManager.registerCommand(pluginName, "changeSPicture", args => {
        const actor = $gameActors.actor(1);
        switch (args["sPictureType"]) {
            case "command":
                commandActorId = actor._actorId;
                break;
            case "damage":
                damageActorId = actor._actorId;
                break;
            case "orgasm":
                orgasmActorId = actor._actorId;
                break;
            default:
                // nop.
        }
    });

    // ランダムID設定
    PluginManager.registerCommand(pluginName, "setRandomId", args => {
        sCommandPictures.randomId = args["randomId"] || 0;
        sDamagePictures.randomId = args["randomId"] || 0;
        sTurnEndPictures.randomId = args["randomId"] || 0;
    });

    //-------------------------------------------------------------------------
    // Game_Actor
    //
    // performDamage関数の処理をパラメータで分岐できるようにします。(SVでは使用不可)

    Game_Actor.prototype.performDamageEx = function(type, target, value, changeSPictureFlag) {
        // SVでは何も起こらない
        if ($gameSystem.isSideView()) {
            return;
        }

        // ダメージ表現
        switch (type) {
            case "normal":
                $gameScreen.startShake(5, 5, 10);
                AudioManager.playSe({
                    name: "Damage4",
                    pan: 0,
                    pitch: 100,
                    volume: 90,
                });
                break;
            case "hard":
                $gameScreen.startShake(5, 5, 30);
                AudioManager.playSe({
                    name: "Damage4",
                    pan: 0,
                    pitch: 70,
                    volume: 90,
                });
                break;
            default:
                // nop.
        }

        // 立ち絵変更の有無
        if (changeSPictureFlag) {
            damageActorId = this._actorId;
        }

        // ダメージ処理
        const resValue = ((value) => {
            if (!isNaN(value)) {
                return parseInt(value);
            }
            const d = (num) => { // eslint-disable-line no-unused-vars
                return Math.floor(Math.random() * 2 * num - num);
            };
            const result = Math.max(Math.ceil(eval(value)), 0);
            return isNaN(result) ? 0 : result;
        })(value);
        if (this.isAlive()) {
            let fmt;
            switch (target) {
                case "hp":
                    this.gainHp(-resValue);
                    BattleManager._logWindow.popupDamage(this);
                    fmt = TextManager.actorDamage.format("こはる", resValue);
                    BattleManager._logWindow.addText(fmt);
                    break;
                case "tp":
                    this.gainTp(resValue);
                    BattleManager._logWindow.popupDamage(this);
                    fmt = TextManager.actorRape.format("こはる", resValue);
                    BattleManager._logWindow.addText(fmt);
                    break;
                default:
                    // nop.
            }
            if (this.isDead()) {
                this.performCollapse();
            }
        }
    };

    //-------------------------------------------------------------------------
    // SPictureList
    //
    // 立ち絵リストクラスを追加定義します。

    function SPictureList() {
        this.initialize(...arguments);
    };
    
    SPictureList.prototype.initialize = function(sPictures) {
        this.sPictures = sPictures;
        this.randomId = 0;
    };
    
    SPictureList.prototype.search = function() {
        const actorId = 1;
        const actorStates = $gameActors.actor(actorId)._states;
    
        const matchStateCase = function(picture) {
            const res = picture.stateCase.filter((x) => {
                return actorStates.indexOf(Number(x.id)) !== -1;
            });
            return res.length > 0 && res.length === picture.stateCase.length;
        };
        const matchSwitchCase = function(picture) {
            const res = picture.switchCase.filter((x) => {
                return $gameSwitches.value(Number(x.id));
            });
            return res.length > 0 && res.length === picture.switchCase.length;
        };
        const matchVariableCase = function(picture) {
            const res = picture.variableCase.filter((x) => {
                return String(x.type) === "equal"
                        && $gameVariables.value(Number(x.id)) == Number(x.value)
                    || String(x.type) === "higher"
                        && $gameVariables.value(Number(x.id)) >= Number(x.value)
                    || String(x.type) === "lower"
                        && $gameVariables.value(Number(x.id)) <= Number(x.value);
            });
            return res.length > 0 && res.length === picture.variableCase.length;
        };

        const randomSelect = function(pictures) {
            const ratingMax = Math.max(...pictures.map(a => a.rating));
            pictures = pictures.filter(x => x.rating === ratingMax);
            if (pictures.length === 1) {
                this.randomId = 0;
                return pictures[0];
            }
            const oldId = parseInt(this.randomId);
            let newId = parseInt(this.randomId);
            while (newId === oldId) {
                newId = Math.randomInt(pictures.length);
            }
            this.randomId = newId;
            return pictures[this.randomId];
        }.bind(this);

        let filteredPictures = [];
    
        // 1. ステートID、スイッチID、変数条件全てに一致するもの
        filteredPictures = this.sPictures.filter(function(picture) {
            return matchStateCase(picture)
                && matchSwitchCase(picture)
                && matchVariableCase(picture);
        });
        if (filteredPictures.length) return randomSelect(filteredPictures);
    
        // 2. ステートID、スイッチID両方に一致するもの
        filteredPictures = this.sPictures.filter(function(picture) {
            return matchStateCase(picture)
                && matchSwitchCase(picture)
                && picture.variableCase.length === 0;
        });
        if (filteredPictures.length) return randomSelect(filteredPictures);
    
        // 3. ステートID、変数条件両方に一致するもの
        filteredPictures = this.sPictures.filter(function(picture) {
            return matchStateCase(picture)
                && picture.switchCase.length === 0
                && matchVariableCase(picture);
        });
        if (filteredPictures.length) return randomSelect(filteredPictures);
    
        // 4. ステートIDのみ一致するもの
        filteredPictures = this.sPictures.filter(function(picture) {
            return matchStateCase(picture)
                && picture.switchCase.length === 0
                && picture.variableCase.length === 0;
        });
        if (filteredPictures.length) return randomSelect(filteredPictures);
    
        // 5. スイッチID、変数条件両方に一致するもの
        filteredPictures = this.sPictures.filter(function(picture) {
            return picture.stateCase.length === 0
                && matchSwitchCase(picture)
                && matchVariableCase(picture);
        });
        if (filteredPictures.length) return randomSelect(filteredPictures);
    
        // 6. スイッチIDのみ一致するもの
        filteredPictures = this.sPictures.filter(function(picture) {
            return picture.stateCase.length === 0
                && matchSwitchCase(picture)
                && picture.variableCase.length === 0;
        });
        if (filteredPictures.length) return randomSelect(filteredPictures);
    
        // 7. 変数条件のみ一致するもの
        filteredPictures = this.sPictures.filter(function(picture) {
            return picture.stateCase.length === 0
                && picture.switchCase.length === 0
                && matchVariableCase(picture);
        });
        if (filteredPictures.length) return randomSelect(filteredPictures);
    
        // 8. 条件なし (ステートID、スイッチID、変数条件全て設定なし)
        filteredPictures = this.sPictures.filter(function(picture) {
            return picture.stateCase.length === 0
                && picture.switchCase.length === 0
                && picture.variableCase.length === 0;
        });
        if (filteredPictures.length) return randomSelect(filteredPictures);
    };

    // 全立ち絵を格納したオブジェクトを作成
    const parseJsonParam = function(paramName) {
        return new SPictureList(JSON.parse(
            JSON.stringify(
                String(parameters[paramName] || "[]"),
                function (key, value) {
                    try {
                        return JSON.parse(value);
                    } catch (e) {
                        return value;
                    }
                }
            )
        ));
    };

    const sCommandPictures = parseJsonParam("sCommandPictures");
    const sDamagePictures = parseJsonParam("sDamagePictures");
    const sTurnEndPictures = parseJsonParam("sTurnEndPictures");

    //-------------------------------------------------------------------------
    // Sprite_SPicture
    //
    // 立ち絵を表示するための独自のスプライトを追加定義します。

    function Sprite_SPicture() {
        this.initialize.apply(this, arguments);
    }

    Sprite_SPicture.prototype = Object.create(Sprite.prototype);
    Sprite_SPicture.prototype.constructor = Sprite_SPicture;

    Sprite_SPicture.prototype.initialize = function() {
        Sprite.prototype.initialize.call(this);
        this.bitmap = null;
        this.opacity = 0;
        this.opening = false;
        this.closing = false;
        this.originX = 0;
        this.originY = 0;
        this.showing = false;
        this.animationFrame = {
            "yes":            24,
            "yesyes":         48,
            "no":             24,
            "noslow":         48,
            "jump":           24,
            "jumpjump":       48,
            "jumploop":       48,
            "shake":          1,
            "shakeloop":      1,
            "runleft":        1,
            "runright":       1,
            "damage":         1,
            "harddamage":     1,
            "floatrightfast": 12,
            "floatright":     48,
            "floatleftfast":  12,
            "floatleft":      48,
            "noslowloop":     96,
            "breathing":      96,
            "breathing2":     96,
            "stepleft":       24,
            "stepright":      24,
            "headdown":       12,
            "none":           0
        };
    };

    Sprite_SPicture.prototype.refresh = function(sPicture) {
        this.setPicture(sPicture);
        this.showing = false;
        let calcScaleX = Number(sPicture.scaleX);
        let calcScaleY = Number(sPicture.scaleY);
        // 画像が読み込まれたあとに実行
        this.bitmap.addLoadListener(function() {
            if (Number(sPicture.origin) !== 1 && String(sPicture.origin) !== "center") {
                // 左上原点
                this.x = Number(sPicture.x);
                this.y = Number(sPicture.y);
                this.originX = Number(sPicture.x);
                this.originY = Number(sPicture.y);
            } else {
                // 中央原点
                this.x = Number(sPicture.x) - (this.width * calcScaleX / 100) / 2;
                this.y = Number(sPicture.y) - (this.height * calcScaleY / 100) / 2;
                this.originX = Number(sPicture.x) - (this.width * calcScaleX / 100) / 2;
                this.originY = Number(sPicture.y) - (this.height * calcScaleY / 100) / 2;
            }
            // 切替効果
            if (this.opacity === 0) {
                // nop.
            }
            this.scale.x = calcScaleX / 100;
            this.scale.y = calcScaleY / 100;
            this.showing = true;
        }.bind(this));
    };

    Sprite_SPicture.prototype.setPicture = function(sPicture) {
        this.bitmap = null;
        this.bitmap = ImageManager.loadPicture(sPicture.imageName);
    };

    Sprite_SPicture.prototype.fadeIn = function() {
        if (!this.showing) return;

        if (this.opacity >= 255) {
            this.opening = false;
            this.opacity = 255;
            return;
        }

        this.opening = true;
        this.closing = false;
        this.opacity += 24;
    };

    Sprite_SPicture.prototype.fadeOut = function() {
        if (this.opacity <= 0) {
            this.closing = false;
            return;
        }

        this.closing = true;
        this.opacity -= 24;
    };

    Sprite_SPicture.prototype.animation = function(sMotion, animationCount) {
        if (!this.showing) return animationCount;

        if (sMotion == "yes") {
            if (animationCount > 12) {
                this.y += 2;
            } else {
                this.y -= 2;
            }
            animationCount -= 1;
        }

        if (sMotion == "yesyes") {
            if (animationCount > 36) {
                this.y += 2;
            } else if (animationCount > 24) {
                this.y -= 2;
            } else if (animationCount > 12) {
                this.y += 2;
            } else {
                this.y -= 2;
            }
            animationCount -= 1;
        }

        if (sMotion == "no") {
            if (animationCount > 18) {
                this.x += 2;
            } else if (animationCount > 6) {
                this.x -= 2;
            } else {
                this.x += 2;
            }
            animationCount -= 1;
        }

        if (sMotion == "noslow") {
            if (animationCount > 36) {
                this.x += 1;
            } else if (animationCount > 12) {
                this.x -= 1;
            } else {
                this.x += 1;
            }
            animationCount -= 1;
        }

        if (sMotion == "jump") {
            if (animationCount > 12) {
                this.y -= 2;
            } else {
                this.y += 2;
            }
            animationCount -= 1;
        }

        if (sMotion == "jumpjump") {
            if (animationCount > 36) {
                this.y -= 2;
            } else if (animationCount > 24) {
                this.y += 2;
            } else if (animationCount > 12) {
                this.y -= 2;
            } else {
                this.y += 2;
            }
            animationCount -= 1;
        }

        if (sMotion == "jumploop") {
            if (animationCount > 36) {
                this.y -= 2;
            } else if (animationCount > 24) {
                this.y += 2;
            }
            animationCount -= 1;
            if (animationCount == 0) animationCount = this.animationFrame["jumploop"];
        }

        if (sMotion == "shake") {
            if (animationCount <= 2) {
                this.x -= 2;
                animationCount += 1;
            } else if (animationCount <= 4) {
                this.y -= 2;
                animationCount += 1;
            } else if (animationCount <= 6) {
                this.x += 4;
                this.y += 4;
                animationCount += 1;
            } else if (animationCount <= 8) {
                this.y -= 2;
                animationCount += 1;
            } else if (animationCount == 9) {
                this.x -= 2;
                animationCount += 1;
            } else if (animationCount == 10) {
                this.x -= 2;
                animationCount = 0;
            }
        }

        if (sMotion == "shakeloop") {
            if (animationCount <= 2) {
                this.x -= 1;
                animationCount += 1;
            } else if (animationCount <= 4) {
                this.y -= 1;
                animationCount += 1;
            } else if (animationCount <= 6) {
                this.x += 2;
                this.y += 2;
                animationCount += 1;
            } else if (animationCount <= 8) {
                this.y -= 1;
                animationCount += 1;
            } else if (animationCount <= 10) {
                this.x -= 1;
                animationCount += 1;
            }
            if (animationCount > 10) animationCount = 1;
        }

        if (sMotion == "runleft") {
            this.x -= 16;
            if (this.x < -2000) animationCount = 0;
        }

        if (sMotion == "runright") {
            this.x += 16;
            if (this.x > 2000) animationCount = 0;
        }

        if (sMotion == "damage") {
            if (animationCount <= 2) {
                this.x -= 4;
                animationCount += 1;
            } else if (animationCount <= 4) {
                this.y -= 4;
                animationCount += 1;
            } else if (animationCount <= 6) {
                this.x += 8;
                this.y += 8;
                animationCount += 1;
            } else if (animationCount <= 8) {
                this.y -= 4;
                animationCount += 1;
            } else if (animationCount == 9) {
                this.x -= 4;
                animationCount += 1;
            } else if (animationCount == 10) {
                this.x -= 4;
                animationCount = 0;
            }
        }

        if (sMotion == "harddamage") {
            if (animationCount <= 2) {
                this.x -= 12;
                animationCount += 1;
            } else if (animationCount <= 4) {
                this.y -= 12;
                animationCount += 1;
            } else if (animationCount <= 6) {
                this.x += 24;
                this.y += 24;
                animationCount += 1;
            } else if (animationCount <= 8) {
                this.y -= 12;
                animationCount += 1;
            } else if (animationCount == 9) {
                this.x -= 12;
                animationCount += 1;
            } else if (animationCount == 10) {
                this.x -= 12;
                animationCount = 0;
            }
        }

        if (sMotion == "floatrightfast") {
            if (animationCount == 12) {
                this.x += 22;
            } else {
                this.x -= 2;
            }
            animationCount -= 1;
        }

        if (sMotion == "floatright") {
            if (animationCount == 48) {
                this.x += 47;
            } else {
                this.x -= 1;
            }
            animationCount -= 1;
        }

        if (sMotion == "floatleftfast") {
            if (animationCount == 12) {
                this.x -= 22;
            } else {
                this.x += 2;
            }
            animationCount -= 1;
        }

        if (sMotion == "floatleft") {
            if (animationCount == 48) {
                this.x -= 47;
            } else {
                this.x += 1;
            }
            animationCount -= 1;
        }

        if (sMotion == "noslowloop") {
            if (animationCount > 72) {
                this.x += 0.25;
            } else if (animationCount > 24) {
                this.x -= 0.25;
            } else {
                this.x += 0.25;
            }
            animationCount -= 1;
            if (animationCount == 0) animationCount = this.animationFrame["noslowloop"];
        }

        if (sMotion == "breathing") {
            if (animationCount > 72) {
                this.y += 0.5;
            } else if (animationCount > 48) {
                this.y -= 0.5;
            } else {
            }
            animationCount -= 1;
            if (animationCount == 0) animationCount = this.animationFrame["breathing"];
        }

        if (sMotion == "breathing2") {
            if (animationCount > 48) {
                // this.anchor.y = 1;
                this.y -= this.height * 0.0003;
                this.scale.y += 0.0003;
            } else {
                // this.anchor.y = 1;
                this.y += this.height * 0.0003;
                this.scale.y -= 0.0003;
            }
            animationCount -= 1;
            if (animationCount == 0) animationCount = this.animationFrame["breathing2"];
        }

        if (sMotion == "stepleft") {
            if (animationCount > 12) {
                this.x -= 2;
            } else {
                this.x += 2;
            }
            animationCount -= 1;
        }

        if (sMotion == "stepright") {
            if (animationCount > 12) {
                this.x += 2;
            } else {
                this.x -= 2;
            }
            animationCount -= 1;
        }

        if (sMotion == "headdown") {
            this.y += 2;
            animationCount -= 1;
        }

        return animationCount;
    };

    //-------------------------------------------------------------------------
    // BattleManager

    BattleManager.isPhase = function() {
        return this._phase;
    };

    //-------------------------------------------------------------------------
    // Game_Battler

    const _Game_Battler_performDamage = Game_Battler.prototype.performDamage;
    Game_Battler.prototype.performDamage = function() {
        _Game_Battler_performDamage.apply(this, arguments);
        damageActorId = this._actorId;
    };

    //-------------------------------------------------------------------------
    // Scene_Battle

    const _Scene_Battle_initialize = Scene_Battle.prototype.initialize;
    Scene_Battle.prototype.initialize = function() {
        _Scene_Battle_initialize.apply(this, arguments);
        this.sPicture = null;
        this.animationCount = 0;
    };

    const _Scene_Battle_update = Scene_Battle.prototype.update;
    Scene_Battle.prototype.update = function() {
        _Scene_Battle_update.apply(this, arguments);
        this.updateSPicture();
    };

    Scene_Battle.prototype.updateSPicture = function() {
        let isPhase = BattleManager.isPhase();
        let refreshFlag = false;

        // コマンド選択前
        if (isPhase === "start" || commandActorId) {
            commandActorId = null;
            const result = sCommandPictures.search();
            if (result && this.sPicture !== result) {
                this.sPicture = result;
                refreshFlag = true;
                this.animationCount = this.spSprite.animationFrame[this.sPicture.motion];
            }
        }

        // 被ダメージ時
        if (damageActorId) {
            damageActorId = null;
            const result = sDamagePictures.search();
            if (result && this.sPicture !== result) {
                this.sPicture = result;
                refreshFlag = true;
            }
            this.animationCount = this.spSprite.animationFrame[this.sPicture.motion];
        }

        // 絶頂時
        if (orgasmActorId) {
            orgasmActorId = null;
            const result = sTurnEndPictures.search();
            if (result && this.sPicture !== result) {
                this.sPicture = result;
                refreshFlag = true;
                this.animationCount = this.spSprite.animationFrame[this.sPicture.motion];
            }
        }

        // 立ち絵初期化
        if (refreshFlag) {
            refreshFlag = false;
            this.spSprite.refresh(this.sPicture);
        }

        // フェード処理
        if (this.sPicture) {
            this.spSprite.fadeIn();
        } else {
            this.spSprite.fadeOut();
        }

        // 立ち絵アニメーション
        if (this.animationCount > 0) {
            this.animationCount = this.spSprite.animation(this.sPicture.motion, this.animationCount);
        }
    };

    const _Scene_Battle_createSpriteset = Scene_Battle.prototype.createSpriteset;
    Scene_Battle.prototype.createSpriteset = function() {
        _Scene_Battle_createSpriteset.apply(this, arguments);
        this.createSPictureSpriteset();
    };

    Scene_Battle.prototype.createSPictureSpriteset = function() {
        this.spSprite = new Sprite_SPicture();
        this._spriteset._battleField.addChild(this.spSprite);
    };

    const _Scene_Battle_createStatusWindow = Scene_Battle.prototype.createStatusWindow;
    Scene_Battle.prototype.createStatusWindow = function() {
        _Scene_Battle_createStatusWindow.apply(this, arguments);
        BattleManager._statusWindow = this._statusWindow;
    };

    //-------------------------------------------------------------------------
    // Scene_Battle
    Scene_Battle.prototype.isRightInputMode = function() {
        return false;
    };

    //-------------------------------------------------------------------------
    // Window_BattleStatus

    Window_BattleStatus.prototype.maxCols = function() {
        return 1;
    };

    Window_BattleStatus.prototype.numVisibleRows = function() {
        return 1;
    };

    Window_BattleStatus.prototype.drawItem = function(index) {
        this.drawItemStatus(index);
    };

    Window_BattleStatus.prototype.nameY = function(rect) {
        return this.basicGaugesY(rect) - this.gaugeLineHeight() - 8;
    };

    Window_BattleStatus.prototype.basicGaugesY = function(rect) {
        const bottom = rect.y + rect.height - this.extraHeight();
        const numGauges = $dataSystem.optDisplayTp ? 3 : 2;
        const gaugesHeight = this.gaugeLineHeight() * numGauges;
        return bottom - ImageManager.iconHeight - gaugesHeight - 8;
    };

    Window_BattleStatus.prototype.stateIconX = function(rect) {
        return rect.x + 4;
    };

    Window_BattleStatus.prototype.stateIconY = function(rect) {
        const bottom = rect.y + rect.height - this.extraHeight();
        return bottom - ImageManager.iconHeight;
    };

    Window_BattleStatus.prototype.placeStateIcon = function(actor, x, y) {
        this.drawStateIcon(actor, x, y);
    };

    Window_BattleStatus.prototype.drawStateIcon = function(actor, x, y) {
        this.drawActorIcons(actor, x, y, this.itemWidth() - 8);
    };

    Window_BattleStatus.prototype.refreshContentsDraw = function() {
        this.contents.clear();
        for (const actor of $gameParty.members()) {
            const index = actor.index();
            const rect = this.itemRectWithPadding(index);
            const stateIconX = this.stateIconX(rect);
            const stateIconY = this.stateIconY(rect);
            this.drawStateIcon(actor, stateIconX, stateIconY);
        }
    };

    Window_BattleStatus.prototype.loadWindowskin = function() {
        this.windowskin = null;
    };

    Window_BattleStatus.prototype.drawItemBackground = function(/*index*/) {
        //
    };

    //-------------------------------------------------------------------------
    // Sprite_Gauge

    const _Sprite_Gauge_bitmapWidth = Sprite_Gauge.prototype.bitmapWidth;
    Sprite_Gauge.prototype.bitmapWidth = function() {
        if (SceneManager._scene.constructor.name === "Scene_Battle") {
            return Number(parameters["gaugeWidth"] || 128);
        }
        return _Sprite_Gauge_bitmapWidth.apply(this, arguments);
    };

    //-------------------------------------------------------------------------
    // Game_Actor

    const _Game_Actor_addState = Game_Actor.prototype.addState;
    Game_Actor.prototype.addState = function(stateId) {
        _Game_Actor_addState.call(this, stateId);
        if (BattleManager._statusWindow) {
            BattleManager._statusWindow.refreshContentsDraw();
        }
    };

    const _Game_Actor_removeState = Game_Actor.prototype.removeState;
    Game_Actor.prototype.removeState = function(stateId) {
        _Game_Actor_removeState.call(this, stateId);
        if (BattleManager._statusWindow) {
            BattleManager._statusWindow.refreshContentsDraw();
        }
    };

    //-------------------------------------------------------------------------
    // Window_BattleSkill

    Window_BattleSkill.prototype.show = function() {
        this.selectLast();
        this.showHelpWindow();
        this._helpWindow.clear();
        Window_SkillList.prototype.show.call(this);
    };

    //-------------------------------------------------------------------------
    // ColorManager

    const _ColorManager_hpColor = ColorManager.hpColor;
    ColorManager.hpColor = function(actor) {
        if (actor && actor.hp === 0) {
            return this.deathColor();
        } else {
            return _ColorManager_hpColor.call(this, actor);
        }
    };

    //-------------------------------------------------------------------------
    // Game_Actor
    //
    // TP(EP)ダメージでもポップアップ表示されるようにします

    const _Game_Actor_shouldPopupDamage = Game_Actor.prototype.shouldPopupDamage;
    Game_Actor.prototype.shouldPopupDamage = function() {
        const parentReturn = _Game_Actor_shouldPopupDamage.call(this);
        const result = this._result;
        return parentReturn || result.tpDamage !== 0;
    };
    
    //-------------------------------------------------------------------------
    // Sprite_Damage
    //
    // TP(EP)ダメージ時のポップアップ色を変更します

    Sprite_Damage.prototype.setup = function(target) {
        const result = target.result();
        if (result.missed || result.evaded) {
            this._colorType = 0;
            this.createMiss();
        } else if (result.hpAffected) {
            this._colorType = result.hpDamage >= 0 ? 0 : 1;
            this.createDigits(result.hpDamage);
        } else if (target.isAlive() && result.mpDamage !== 0) {
            this._colorType = result.mpDamage >= 0 ? 2 : 3;
            this.createDigits(result.mpDamage);
        } else if (target.isAlive() && result.tpDamage !== 0) {
            this._colorType = result.tpDamage >= 0 ? 4 : 5;
            this.createDigits(result.tpDamage);
        }
        if (result.critical) {
            this.setupCriticalEffect();
        }
    };

    //-------------------------------------------------------------------------
    // ColorManager

    const _ColorManager_damageColor = ColorManager.damageColor;
    ColorManager.damageColor = function(colorType) {
        switch (colorType) {
            case 4: // TP減少
                return this.textColor(24);
            case 5: // TP増加
                return this.textColor(27);
            default:
                return _ColorManager_damageColor.call(this, colorType);
        }
    };
})();
